<?php
/**
 Plugin Name: FG - Gestión de Stock, Precios y Ofertas
 * Plugin URI: https://fgwordpress.com.ar/
 * Description: Gestiona stock, precios y ofertas. Permite buscar por SKU y filtrar por categoría y tipo de precio. En la versión Pro, puede editar las fechas de Inicio y Fin de las Ofertas. Limpieza automática de ofertas vencidas a medianoche.
 * Version: 2.1
 * Author: Fabián Osvaldo Grasso
 * Author URI: 
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) exit; // seguridad linea 522 y 550

class FG_Gestion_Stock_Precios_Ofertas {

    private $p_flag; 

    public function __construct() {
        add_action('admin_menu', array($this, 'agregar_submenu'));
        add_action('admin_post_fg_actualizar_precios_stock', array($this, 'guardar_datos'));
        add_action('wp_ajax_fg_actualizar_precios_stock', array($this, 'guardar_datos'));        
        add_action('wp_ajax_fg_toggle_featured', array($this, 'ajax_toggle_featured'));        
        add_action('admin_enqueue_scripts', array($this, 'estilos_admin'));
        add_filter('plugin_row_meta', array($this, 'plugin_row_meta'), 10, 4);	
		// Hook del cron de WooCommerce        		   							
		add_action( 'woocommerce_scheduled_sales', [ $this, 'fg_repair_after_wc_sale_cleanup' ], 999 );
		add_action('woocommerce_before_single_product', [ $this,'fg_forzar_limpiar_oferta_vencida' ], 999);	
		
		add_filter('woocommerce_product_get_price', [ $this, 'fg_filtrar_precio_oferta_vencida' ], 20, 2);

    $is_pro = get_option('fg_has_pro', false);

    if ($is_pro) {
        $this->p_flag = true;
    } else {
        // si no es pro → usar trial
        $this->p_flag = fg_is_trial_active();
    }
				
    }

    public function agregar_submenu() {
        add_submenu_page(
            'woocommerce',
            'FG Gestión de Stock, Precios y Ofertas',
            'FG Gestión de Stock, Precios y Ofertas',
            'manage_woocommerce',
            'fg-gestion-stock',
            array($this, 'pagina_principal')
        );
    }

    public function estilos_admin() {
        // estilos encolados
        wp_register_style('fg-admin-style', false);
        wp_enqueue_style('fg-admin-style');
        wp_add_inline_style('fg-admin-style', "
            .fg-card { background: #fff; border-radius: 10px; box-shadow: 0 2px 6px rgba(0,0,0,0.06); padding: 18px; margin-top: 18px; max-width: 1200px; }
            .fg-card h2 { margin-top: 0; color: #007cba; border-bottom: 1px solid #e6eef6; padding-bottom: 8px; }
            .fg-card table.form-table th { width: 160px; font-weight: 600; color: #333; }
            .fg-card input[type='text'], .fg-card input[type='number'], .fg-card input[type='date'], .fg-card select { width: 100%; border-radius: 5px; border: 1px solid #ccc; padding: 6px; }
            .fg-card .button-primary { background-color: #007cba !important; border-color: #007cba !important; border-radius: 6px; font-weight: 600; padding: 8px 14px; }
            .fg-search-box { margin-top: 12px; background: #f8f9fa; border-radius: 8px; padding: 12px; max-width: 1200px; border: 1px solid #e0e0e0; display:flex; gap:10px; align-items:center; }
            .fg-notice { margin-top: 12px; background: #d1fae5; color: #065f46; padding: 8px 12px; border-left: 5px solid #10b981; border-radius: 6px; max-width: 1200px; }
            .fg-table { width: 100%; border-collapse: collapse; margin-top: 12px; }
            .fg-table th, .fg-table td { padding: 8px 10px; border-bottom: 1px solid #eee; text-align: left; vertical-align: middle; }
            .fg-lock { color:#999; }
            .fg-state { font-weight:700; }
            .fg-actions { white-space:nowrap; }
            .fg-updated { background-color: red !important; transition: background-color 2.4s ease; }
        ");

        // Encolar jQuery (WordPress ya lo tiene), y añadir script inline para el comportamiento del admin
        wp_enqueue_script('jquery');
        $admin_js = "jQuery(document).ready(function($) {\n" .
            "    $('#btnLimpiar').on('click', function(e) { e.preventDefault(); $('#sku').val(''); $('#cat').val('all'); $('#tipo').val('all'); $('#miFormulario').submit(); });\n" .
            "    $('.form-actualizar-producto').on('submit', function(e) {\n" .
            "        e.preventDefault();\n" .
            "        var form = $(this);\n" .
            "        var row = form.closest('tr');\n" .
            "        var data = form.serialize();\n" .
            "        var btn = row.find('input[type=\\'submit\\'], button[type=\\'submit\\']').first();\n" .
            "        var oldText = btn.val();\n" .
            "        btn.val('💾').prop('disabled', true);\n" .
            "        row.css('background-color', '#dbdbdb');\n" .
            "        $.post(ajaxurl, data, function(response) {\n" .
            "            if (response && response.success) {\n" .
            "                row.css('background-color', '#fbffa8');\n" .
            "                var celda = row.find('td').eq(8);\n" .
            "                var oferta = row.find('td').eq(4).find('input[type=\\'number\\']');\n" .
            "                var valor_oferta = oferta.val();\n" .
            "                var date_ini = row.find('td').eq(5).find('input[type=\\'date\\']');\n" .
            "                var date_fin = row.find('td').eq(6).find('input[type=\\'date\\']');\n" .
			   "                if (valor_oferta === '' || valor_oferta === null) { date_ini.val(''); date_fin.val(''); celda.text('🔴Inactiva'); } else { celda.text('🟢Activa'); }\n" .
            "            } else { alert('Error al guardar'); }\n" .
            "            btn.val(oldText).prop('disabled', false);\n" .
            "        }).fail(function() { alert('Error al conectar con el servidor'); btn.val(oldText).prop('disabled', false); });\n" .
            "    });\n" .

            // Delegated handler para Toggle Featured (Destacado)
            // uso delegated event para asegurar que funcione aunque los botones sean renderizados dinámicamente
            "    $(document).on('click', '.toggle-featured', function(e) {\n" .
            "        e.preventDefault();\n" .
            "        var btn = $(this);\n" .
            "        var productId = btn.data('id');\n" .
            "        if (!productId) { alert('ID de producto inválido'); return; }\n" .
            "        var oldText = btn.text();\n" .
            "        btn.prop('disabled', true).text('⏳...');\n" .
            "        $.post(ajaxurl, {\n" .
            "            action: 'fg_toggle_featured',\n" .
            "            id: productId\n" .
            "        }, function(response) {\n" .
            "            if (response && response.success) {\n" .
            "                var cls = response.data.class || '';\n" .
            "                var label = response.data.label || (response.data.featured ? '⭐ Destacado' : '❌ No destacado');\n" .
            "                btn.removeClass('btn-warning btn-secondary button-primary button-secondary')\n" .
            "                   .addClass(cls)\n" .
            "                   .text(label);\n" .
            "            } else {\n" .
            "                alert('Error: ' + (response.data || 'No se pudo cambiar el estado'));\n" .
            "                btn.text(oldText);\n" .
            "            }\n" .
            "        }).fail(function() {\n" .
            "            alert('Error de conexión con el servidor');\n" .
            "            btn.text(oldText);\n" .
            "        }).always(function() {\n" .
            "            btn.prop('disabled', false);\n" .
            "        });\n" .
            "    });\n" .
            "});";
        wp_add_inline_script('jquery', $admin_js);
    }

    public function pagina_principal() {
        // === FILTROS ===
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $sku  = isset($_GET['sku']) ? sanitize_text_field(wp_unslash($_GET['sku'])) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $cat  = isset($_GET['cat']) ? sanitize_text_field(wp_unslash($_GET['cat'])) : 'all';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $tipo = isset($_GET['tipo']) ? sanitize_text_field(wp_unslash($_GET['tipo'])) : 'all';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $pagina_actual = isset($_GET['pg']) ? max(1, intval($_GET['pg'])) : 1;
        $por_pagina = 50;

        echo '<div class="wrap">';
        echo '<h1>FG Gestión de Stock, Precios y Ofertas v2.1 PRO</h1>';
		if ( $this->p_flag ) {
		echo '<h2 style="color: red; font-size: 16px;">Versión PRO Trial por 15 días. Luego de los 15 días se activará la versión Free...<a href="https://fgwordpress.com.ar/" target="_blank" style="margin-left: 20px; color: blue; text-decoration: none;">Obtener Versión Pro !!!</a></h2>';
		} else {
		echo '<h2 style="color: red; font-size: 16px;">Versión PRO Trial Caducada. La Versión Free fue activada!!!<a href="https://fgwordpress.com.ar/" target="_blank" style="margin-left: 20px; color: blue; text-decoration: none;">Obtener Versión Pro !!!</a></h2>';
		}

        // === FILTROS VISUALES ===
        echo '<div class="fg-search-box">';
        echo '<form id="miFormulario" method="get" action="">';
        echo '<input type="hidden" name="page" value="fg-gestion-stock">';
        echo '<label for="sku"><strong>Buscar por SKU:</strong></label> ';
        echo '<input type="text" name="sku" id="sku" value="'.esc_attr($sku).'" placeholder="Ej: 4512097">';

        $terms = get_terms(array('taxonomy' => 'product_cat', 'hide_empty' => false));
        echo '<label for="cat" style="margin-left: 10px;"><strong>Categoría:</strong></label> ';
		if ($this->p_flag){echo '<select name="cat" id="cat">';} else {echo '<select name="cat" id="cat" disabled>';}       
        echo '<option value="all">Todas</option>';
        if (!is_wp_error($terms)) {
            foreach ($terms as $t) {
                echo '<option value="'.esc_attr($t->slug).'" '.selected($cat, $t->slug, false).'>'.esc_html($t->name).'</option>';
            }
        }
        echo '</select>';

        echo '<label for="tipo" style="margin-left: 10px;"><strong>Tipo de precio:</strong></label> ';
		if ($this->p_flag){echo '<select name="tipo" id="tipo">';} else {echo '<select name="tipo" id="tipo" disabled>';}               
        $tipos = array('all' => 'Todos', 'normal' => 'Precios normales', 'rebajado' => 'Precios rebajados');
        foreach ($tipos as $k => $v) {
            echo '<option value="'.esc_attr($k).'" '.selected($tipo, $k, false).'>'.esc_html($v).'</option>';
        }
        echo '</select>';

        echo '<input type="submit" class="button button-primary" style="background-color: red !important; color: white; margin-left: 10px;" value="Filtrar">';
        echo '<input type="submit" class="button button-primary" style="background-color: yellow !important; color: black; margin-left: 10px;" id="btnLimpiar" value="<<">';

        echo '</form>';
        echo '</div>'; // fg-search-box

        // === OBTENER PRODUCTOS ===
        $args = array(
            'post_type'      => array('product', 'product_variation'),
            'posts_per_page' => -1,
            'tax_query'      => isset($tax_query) ? $tax_query : array(),
        );

        if ($sku) {
    $product_id = wc_get_product_id_by_sku( $sku );

    if ($product_id) {
        $args['post__in'] = array($product_id);
    } else {
        $args['post__in'] = array(0);
    }
}				

        // Si se filtró por categoría
        if ($cat !== 'all') {
            $tax_query = array(
                array(
                    'taxonomy' => 'product_cat',
                    'field'    => 'slug',
                    'terms'    => array($cat),
                ),
            );
            $args['tax_query'] = $tax_query;
        }

        $query = new WP_Query($args);
        $productos = $query->posts;

        // === FILTRAR POR TIPO (normal / rebajado) ===
        $productos_filtrados = array();
        foreach ($productos as $p) {
            $product = wc_get_product($p->ID);
            if (!$product) continue;

            $sale = $product->get_sale_price();

            if ($tipo === 'normal' && ($sale !== '' && $sale !== null)) continue;
            if ($tipo === 'rebajado' && ($sale === '' || $sale === null)) continue;

            $productos_filtrados[] = $p;
        }

        // === PAGINACIÓN (anidar en array_slice) ===
        $total_filtrados = count($productos_filtrados);
        $por_pagina = 50;
        $total_paginas = max(1, ceil($total_filtrados / $por_pagina));
        $pagina_actual = min($pagina_actual, $total_paginas);

        $offset = ($pagina_actual - 1) * $por_pagina;
        $productos_mostrar = array_slice($productos_filtrados, $offset, $por_pagina);

        echo '<div class="fg-card">';
        if (empty($productos)) {
            echo '<div class="fg-notice">No se encontraron productos.</div>';
            echo '</div></div>';
            return;
        }

        // === TABLA ===
        echo '<table class="fg-table">';
        echo '<thead><tr><th>SKU</th><th>Producto</th><th>Categoría</th><th>Precio Normal</th><th>Precio Rebajado</th><th>Fecha de Inicio</th><th>Fecha Final</th><th>Stock</th><th>Oferta</th><th>Estado</th><th>Acción</th></tr></thead>';
        echo '<tbody>';

        foreach ($productos_mostrar as $p) {
            $product = wc_get_product($p->ID);
            if (!$product) continue;

            $sale = $product->get_sale_price();
            $regular = $product->get_regular_price();
			$precio_activo = $product->get_price();
            $fecha_from = get_post_meta($p->ID, '_sale_price_dates_from', true);
            $fecha_to = get_post_meta($p->ID, '_sale_price_dates_to', true);			
            $stock = $product->get_stock_quantity();
            $sku_val = $product->get_sku();			            
			
			$hoy = strtotime( current_time('Y-m-d') );
            $estado = 'Inactiva';
            if ($sale !== '' && $sale !== null) {
                if ($fecha_to && $precio_activo == $regular) $estado = 'Vencida';
                else $estado = 'Activa';
            }

            if ($tipo === 'normal' && ($sale !== '' && $sale !== null)) continue;
            if ($tipo === 'rebajado' && ($sale === '' || $sale === null)) continue;

            $cats = wp_get_post_terms($p->ID, 'product_cat', array('fields' => 'names'));
            $cat_names = implode(', ', $cats);

            echo '<tr id="row-'.esc_attr($p->ID).'">';
            echo '<form class="form-actualizar-producto" data-product-id="'.esc_attr($p->ID).'">';
			wp_nonce_field('mi_accion_segura', 'mi_nonce');
            echo '<input type="hidden" name="action" value="fg_actualizar_precios_stock">';
            echo '<input type="hidden" name="product_id" value="'.esc_attr($p->ID).'">';

            echo '<td>'.esc_html($sku_val).'</td>';
            echo '<td>'.esc_html($product->get_name()).'</td>';
            echo '<td>'.esc_html($cat_names).'</td>';

            echo '<td><input type="number" style="width:90px;" name="precio_regular" value="'.esc_attr($regular).'"></td>';
            echo '<td><input type="number" style="width:90px;" name="precio_oferta" value="'.esc_attr($sale).'"></td>';

            // === Fechas ===
            $from_val = '';
            if (!empty($fecha_from)) {
                $timestamp_from = intval($fecha_from);
                if ($timestamp_from > 0) {
                    $from_val = date_i18n('Y-m-d', $timestamp_from);
                }
            }
            $to_val = '';
            if (!empty($fecha_to)) {
                $timestamp_to = intval($fecha_to);
                if ($timestamp_to > 0) {
                    $to_val = date_i18n('Y-m-d', $timestamp_to - DAY_IN_SECONDS);
                }
            }

            // Atributos para inputs de fecha: usar disabled() y title seguro si la función no está activa
            $disabled_attr = $this->p_flag ? '' : disabled( true, true, false );
            $title_attr = $this->p_flag ? '' : ' title="Función deshabilitada"';

            echo '<td><input type="date" name="fecha_inicio" value="' . esc_attr($from_val) . '" ' . esc_attr($disabled_attr) . ' ' . esc_attr($title_attr) . '></td>';
echo '<td><input type="date" name="fecha_fin" value="' . esc_attr($to_val) . '" ' . esc_attr($disabled_attr) . ' ' . esc_attr($title_attr) . '></td>';

            echo '<td><input type="number" style="width:60px;" name="stock" value="'.esc_attr($stock).'"></td>';

            $icon = '⚪';
            if ($estado === 'Activa') $icon = '🟢';
            if ($estado === 'Vencida') $icon = '🔴';
            if ($estado === 'Inactiva') $icon = '🔴';

            echo '<td>'.esc_html($icon . ' ' . $estado).'</td>';

            // === Columna Featured (botón) ===
            // Usamos $product->is_featured() para obtener el estado real de WooCommerce            
            $is_featured = has_term('featured', 'product_visibility', $p->ID);
            $featured_label = $is_featured ? '⭐ Destacado' : '❌ No destacado';
            $featured_class = $is_featured ? 'btn-warning' : 'btn-secondary';

            echo '<td><button type="button" class="toggle-featured ' . esc_attr($featured_class) . '" data-id="' . esc_attr($p->ID) . '">' . esc_html($featured_label) . '</button></td>';

            echo '<td><input type="submit" class="button button-primary" value="💾"></td>';
            echo '</form>';
            echo '</tr>';
        }

        echo '</tbody></table>';

        // === PAGINACIÓN ===
        if ($total_paginas > 1) {
            echo '<nav aria-label="Paginación de productos" style="margin-top:20px;">';
            echo '<ul class="pagination justify-content-center">';

            // Botón "Anterior"
            if ($pagina_actual > 1) {
                $prev = $pagina_actual - 1;
                $link_prev = esc_url(admin_url('admin.php?page=fg-gestion-stock&pg=' . $prev));
                if (!empty($sku))  $link_prev .= '&sku=' . urlencode($sku);
                if ($cat !== 'all') $link_prev .= '&cat=' . urlencode($cat);
                if ($tipo !== 'all') $link_prev .= '&tipo=' . urlencode($tipo);

                echo '<li class="page-item"><a class="page-link" href="'.esc_url($link_prev).'">&laquo; Anterior</a></li>';
            } else {
                echo '<li class="page-item disabled"><span class="page-link">&laquo; Anterior</span></li>';
            }

            for ($i = 1; $i <= $total_paginas; $i++) {
                $link = admin_url('admin.php?page=fg-gestion-stock&pg=' . $i);
                if (!empty($sku))  $link .= '&sku=' . urlencode($sku);
                if ($cat !== 'all') $link .= '&cat=' . urlencode($cat);
                if ($tipo !== 'all') $link .= '&tipo=' . urlencode($tipo);

                $active = ($i == $pagina_actual) ? 'active' : '';
                echo '<li class="page-item '.esc_attr($active).'"><a class="page-link" href="'.esc_url($link).'">'.esc_html($i).'</a></li>';
            }

            // Botón "Siguiente"
            if ($pagina_actual < $total_paginas) {
                $next = $pagina_actual + 1;
                $link_next = admin_url('admin.php?page=fg-gestion-stock&pg=' . $next);
                if (!empty($sku))  $link_next .= '&sku=' . urlencode($sku);
                if ($cat !== 'all') $link_next .= '&cat=' . urlencode($cat);
                if ($tipo !== 'all') $link_next .= '&tipo=' . urlencode($tipo);

                echo '<li class="page-item"><a class="page-link" href="'.esc_url($link_next).'">Siguiente &raquo;</a></li>';
            } else {
                echo '<li class="page-item disabled"><span class="page-link">Siguiente &raquo;</span></li>';
            }

            echo '</ul></nav>';
        }

        echo '</div>'; // fg-card
        echo '</div>'; // wrap

        echo '<style>
.pagination {
    display: flex !important;
    justify-content: center !important;
    flex-wrap: wrap;
    list-style: none;
    padding-left: 0;
    margin: 20px 0;
}
.page-item {
    margin: 2px;
}
.page-item .page-link {
    display: block;
    padding: 6px 12px;
    color: #007bff;
    background-color: #fff;
    border: 1px solid #dee2e6;
    border-radius: .25rem;
    text-decoration: none;
    transition: all 0.2s;
}
.page-item .page-link:hover {
    background-color: #007bff;
    color: #fff;
}
.page-item.active .page-link {
    background-color: #007bff;
    color: #fff;
    border-color: #007bff;
}
.page-item.disabled .page-link {
    color: #6c757d;
    background-color: #fff;
    border-color: #dee2e6;
    cursor: not-allowed;
}
</style>';
    }


//////////////////////////////////////////Guardar datos///////////////////////////////////////////////////////
public function guardar_datos() {
	if (!current_user_can('manage_woocommerce')) wp_die('Sin permisos');
	// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
	if (!isset($_POST['mi_nonce']) || !wp_verify_nonce(wp_unslash($_POST['mi_nonce']), 'mi_accion_segura')) {
    	wp_die('Acceso no permitido');
	}
	$product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
		
	// Cargar el array POST des-slasheado SOLO una vez
	$post_data = wp_unslash($_POST);
		
	$raw_precio_regular = isset($post_data['precio_regular']) ? $post_data['precio_regular'] : null;
	$precio_regular = $raw_precio_regular !== null ? wc_format_decimal($raw_precio_regular, wc_get_price_decimals()) : '';

	$raw_precio_oferta = isset($post_data['precio_oferta']) ? $post_data['precio_oferta'] : null;
	$precio_oferta = $raw_precio_oferta !== null ? wc_format_decimal($raw_precio_oferta, wc_get_price_decimals()) : '';
	
	$raw_fecha_inicio = isset($post_data['fecha_inicio']) ? $post_data['fecha_inicio'] : null;
	$fecha_inicio = !empty($raw_fecha_inicio) ? strtotime(sanitize_text_field($raw_fecha_inicio)) : '';
         
	$raw_fecha_fin = isset($post_data['fecha_fin']) ? $post_data['fecha_fin'] : null;
	$fecha_fin = !empty($raw_fecha_fin) ? strtotime(sanitize_text_field($raw_fecha_fin)) : '';
        
    $stock = isset($_POST['stock']) ? intval($_POST['stock']) : '';	

    if ($product_id) {
		$product = wc_get_product( $product_id );
        // Regular
        if ($precio_regular !== '') {
        	if ( $product ) {
            	$product->set_regular_price( (string) $precio_regular );
            } else {
                update_post_meta( $product_id, '_regular_price', $precio_regular );
				update_post_meta( $product_id, '_price', $precio_regular );
            }
            } else {
                if ( $product ){
					$product->set_regular_price( '' );
				}
                else {
					update_post_meta( $product_id, '_regular_price', '' );
				}
            }

            // Oferta
            // $has_valid_sale=true (si hay oferta) o false (si no hay oferta)
            $has_valid_sale = ($precio_oferta !== '' && $precio_oferta != 0 && $precio_oferta < $precio_regular);
		
            if ( $has_valid_sale ) {
                if ( $product ){
					$product->set_sale_price( (string) $precio_oferta );
					$product->set_price( (string) $precio_oferta );					
				}
                else {
					update_post_meta( $product_id, '_sale_price', $precio_oferta );
					update_post_meta( $product_id, '_price', $precio_oferta );
				}
            } else {
                if ( $product ) {
                    $product->set_sale_price( '' );
                    $product->set_date_on_sale_from( '' );
                    $product->set_date_on_sale_to( '' );
					$product->set_price( (string) $precio_regular );
                } else {
                    delete_post_meta($product_id, '_sale_price' );
                    delete_post_meta($product_id, '_sale_price_dates_from');
                    delete_post_meta($product_id, '_sale_price_dates_to');
					update_post_meta( $product_id, '_price', $precio_regular );
                }
            }            

            // Stock
            if ( $stock !== '' ) {
                if ( $product ) {
                    if ( $product->managing_stock() ) {
                        $product->set_stock_quantity( intval($stock) );
                    } else {
                        update_post_meta( $product_id, '_stock', intval($stock) );
                    }
                } else {
                    update_post_meta( $product_id, '_stock', intval($stock) );
                }
            }

            // Fechas (solo si la bandera lo permite)
            if ( $this->p_flag && $product ) {                
			 if ( ! empty( $_POST['fecha_inicio'] ) ) {
    			$raw_fecha_inicio = sanitize_text_field( wp_unslash( $_POST['fecha_inicio'] ) );
    			// Convertir usando la zona horaria del sitio
    			$timezone = new DateTimeZone( wp_timezone_string() );

    			// WooCommerce recomienda establecer el final a las 23:59:59 del día correspondiente
    			$dt_inicio = DateTime::createFromFormat('Y-m-d H:i:s', $raw_fecha_inicio . ' 00:00:00', $timezone);

    			if ( $dt_inicio ) {
        			// Convertir al timestamp correcto (ajustado a zona del sitio)
        			$timestamp_inicio = $dt_inicio->getTimestamp();
        			// WooCommerce espera fecha UTC → usar gmdate ESTÁ BIEN AQUÍ
        			$product->set_date_on_sale_from( gmdate('Y-m-d H:i:s', $timestamp_inicio) );
    			}
			} else {
    			$product->set_date_on_sale_from( '' );
			}                
	if ( ! empty( $_POST['fecha_fin'] ) ) {

    $raw_fecha_final = sanitize_text_field( wp_unslash( $_POST['fecha_fin'] ) );

    $timezone = new DateTimeZone( wp_timezone_string() );

    // Hora fija deseada
    $hora = '23:59:59';

    // Detectar formato de fecha ingresado
    if ( strpos( $raw_fecha_final, '/' ) !== false ) {
        // Formato d/m/Y
        $dt_final = DateTime::createFromFormat( 'd/m/Y H:i:s', $raw_fecha_final . ' ' . $hora, $timezone );
    } else {
        // Formato Y-m-d
        $dt_final = DateTime::createFromFormat( 'Y-m-d H:i:s', $raw_fecha_final . ' ' . $hora, $timezone );
    }

    if ( $dt_final ) {

        // IMPORTANTE: NO convertir a UTC
        // WooCommerce espera timestamp UNIX en local para no modificar el día
        $timestamp = $dt_final->getTimestamp();

        update_post_meta( $product->get_id(), '_sale_price_dates_to', $timestamp );
    }

} else {
    // Vaciar la fecha si no se ingresó nada
    delete_post_meta( $product->get_id(), '_sale_price_dates_to' );
}				

            } elseif ( $this->p_flag ) {
            	$raw_fecha_inicio = isset($_POST['fecha_inicio'])? sanitize_text_field( wp_unslash( $_POST['fecha_inicio'])) : '';

			if ( ! empty( $raw_fecha_inicio ) ) {
    			$fecha_inicio = strtotime($raw_fecha_inicio . ' 00:00:00 ' . get_option('timezone_string'));

    			if ( $fecha_inicio ) { // validar timestamp
        			update_post_meta( $product_id, '_sale_price_dates_from', $fecha_inicio );
   				}
			} else {
    			delete_post_meta( $product_id, '_sale_price_dates_from' );
			}
            $raw_fecha_final = isset($_POST['fecha_fin']) ? sanitize_text_field( wp_unslash( $_POST['fecha_fin'])) : '';

			if ( ! empty( $raw_fecha_final ) ) {
    			$fecha_fin = strtotime($raw_fecha_final . ' 23:59:59 ' . get_option('timezone_string'));    			

    		if ( $fecha_fin ) { // validar timestamp				
        		update_post_meta( $product_id, '_sale_price_dates_to', $fecha_fin );        		
    		}

		} else {
    		delete_post_meta( $product_id, '_sale_price_dates_to' );
		}
        }

            // Guardar producto
            if ( $product ) {
				$product->save();
				// Purga de LiteSpeed para este producto
    			do_action('litespeed_purge_post', $product_id);

    			// Limpiar transients del producto
    			wc_delete_product_transients($product_id);
    			clean_post_cache($product_id);
    			wp_cache_delete($product_id, 'posts');
    			wp_cache_delete($product_id, 'post_meta');

    			// Forzar que WooCommerce regenere la lista global de productos en oferta
    			delete_transient('wc_products_onsale');
    			wp_cache_delete('wc_products_onsale', 'transient');

    			// Actualizar manualmente la lista con este producto si sigue en oferta
    			$precio_regular = $product->get_regular_price();
    			$precio_oferta = $product->get_sale_price();
    			$onsale_ids = wc_get_product_ids_on_sale();

    			if ($precio_oferta !== '' && $precio_oferta != 0 && $precio_oferta < $precio_regular) {
        			if (!in_array($product_id, $onsale_ids)) $onsale_ids[] = $product_id;
    				} else {
        		$onsale_ids = array_diff($onsale_ids, array($product_id));
    			}

    			set_transient('wc_products_onsale', $onsale_ids, DAY_IN_SECONDS);
			}
            
            
            
        }

        if ( defined('DOING_AJAX') && DOING_AJAX ) {
            wp_send_json_success('Producto actualizado');
        } else {
            $sku = get_post_meta($product_id, '_sku', true);
            wp_safe_redirect( admin_url( 'admin.php?page=fg-gestion-stock&sku=' . urlencode( $sku ) . '&actualizado=1' ) );
            exit;
        }
    }


    // === AJAX: toggle featured ===
    public function ajax_toggle_featured() {
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Sin permisos');
        }

        $product_id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        if (!$product_id) {
            wp_send_json_error('ID inválido');
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json_error('Producto no encontrado');
        }

        // Alternar usando la API de WooCommerce para featured
        $current = method_exists($product, 'is_featured') ? $product->is_featured() : ( get_post_meta($product_id, '_featured', true) === 'yes' );
        $new_status = !$current;

        if (method_exists($product, 'set_featured')) {
            $product->set_featured($new_status);
            $product->save();
        } else {
            // Fallback: actualizar meta y term visibility
            update_post_meta($product_id, '_featured', $new_status ? 'yes' : 'no');

            // sincronizar product_visibility taxonomy (añadir/remover term 'featured')
            if (taxonomy_exists('product_visibility')) {
                if ($new_status) {
                    // assign featured term
                    wp_set_object_terms($product_id, 'featured', 'product_visibility', true);
                } else {
                    // remove featured term
                    wp_remove_object_terms($product_id, 'featured', 'product_visibility');
                }
            }
        }
		do_action( 'litespeed_purge_url', home_url('/') );
		do_action( 'litespeed_purge_url', wc_get_page_permalink('shop') );
        // Limpiar caches/transients
        if (function_exists('wc_delete_product_transients')) wc_delete_product_transients($product_id);
        delete_transient('wc_products_onsale');

        // Preparar respuesta
        $response = array(
            'featured' => $new_status,
            'label'    => $new_status ? '⭐ Destacado' : '❌ No destacado',
            'class'    => $new_status ? 'btn-warning' : 'btn-secondary'
        );

        wp_send_json_success($response);
    }

    public function plugin_row_meta($links, $file, $plugin_data, $status) {
        if ($file === plugin_basename(__FILE__)) {
            $info_link = "";
            $links[] = $info_link;
        }
        return $links;
    }
	
	public function limpiar_sale_price_expirado() {
    global $wpdb;

    // Buscar productos cuya fecha final de oferta YA venció
    $productos = $wpdb->get_col("
        SELECT post_id 
        FROM $wpdb->postmeta 
        WHERE meta_key = '_sale_price_dates_to' 
        AND meta_value > 0 
        AND meta_value < UNIX_TIMESTAMP()
    ");

    if (empty($productos)) {
        return;
    }

    foreach ($productos as $id) {

        // Obtener precio regular
        $regular = get_post_meta($id, '_regular_price', true);

        // Borrar precio rebajado y fechas
        update_post_meta($id, '_sale_price', '');
        update_post_meta($id, '_sale_price_dates_from', '');
        update_post_meta($id, '_sale_price_dates_to', '');

        // Restaurar precio normal
        update_post_meta($id, '_price', $regular);
// Limpieza WC interna
        wc_delete_product_transients($id);

        // Cache de objeto / meta
        wp_cache_delete($id, 'product');
        wp_cache_delete($id, 'post_meta');
        wp_cache_delete($id, 'posts');

        // Regenerar el objeto del producto
        $product = wc_get_product($id);
        if ($product) {
            $product->save();
        }
    }

    if ( method_exists( $product, 'sync_sale_price' ) ) {
    $product->sync_sale_price();
}

if ( method_exists( $product, 'sync_regular_price' ) ) {
    $product->sync_regular_price();
}

if ( method_exists( $product, 'sync_price' ) ) {
    $product->sync_price();
}
    // 🔥 FORZAR WooCommerce a regenerar la lista de productos con oferta
    delete_transient('wc_products_onsale');
    wp_cache_delete('wc_products_onsale', 'transient');

    // Por las dudas, limpiar también la consulta de on sale products
    delete_transient('wc_featured_products');
    wp_cache_flush();
    do_action('woocommerce_delete_product_query_transients');
		if ( class_exists( 'LiteSpeed_Cache_API' ) ) {
        // Purga total cuando WooCommerce elimina sales vencidas
        do_action( 'litespeed_purge_all' );
    }
	do_action('woocommerce_scheduled_sales');	
}	

public function fg_repair_after_wc_sale_cleanup() {
    global $wpdb;

    $hoy = time();

    // Productos con oferta vencida
    $productos = $wpdb->get_col("
        SELECT post_id 
        FROM $wpdb->postmeta 
        WHERE meta_key = '_sale_price_dates_to'
        AND meta_value > 0
        AND meta_value < {$hoy}
    ");

    foreach ($productos as $product_id) {

        // Borrar datos de oferta
        delete_post_meta($product_id, '_sale_price');
		delete_post_meta($product_id, '_sale_price_dates_from');
        delete_post_meta($product_id, '_sale_price_dates_to');


        // Restaurar precio regular
        $regular = get_post_meta($product_id, '_regular_price', true);
        update_post_meta($product_id, '_price', $regular);

        // Limpieza WC interna
        wc_delete_product_transients($product_id);

        // Cache de objeto / meta
        wp_cache_delete($product_id, 'product');
        wp_cache_delete($product_id, 'post_meta');
        wp_cache_delete($product_id, 'posts');

        // Regenerar el objeto del producto
        $product = wc_get_product($product_id);
        if ($product) {
            $product->save();
        }
    }

    // 🔥 FORZAR WooCommerce a regenerar la lista de productos con oferta
    delete_transient('wc_products_onsale');
    wp_cache_delete('wc_products_onsale', 'transient');

    // Por las dudas, limpiar también la consulta de on sale products
    delete_transient('wc_featured_products');
    wp_cache_flush();
	if ( class_exists( 'LiteSpeed_Cache_API' ) ) {
        // Purga total cuando WooCommerce elimina sales vencidas
        do_action( 'litespeed_purge_all' );
    }
}
	public function fg_forzar_limpiar_oferta_vencida() {
    global $product;

    $sale_to = get_post_meta($product->get_id(), '_sale_price_dates_to', true);

    if ($sale_to && $sale_to < time()) {
        // Oferta vencida → limpiar
        update_post_meta($product->get_id(), '_sale_price', '');
        update_post_meta($product->get_id(), '_sale_price_dates_from', '');
        update_post_meta($product->get_id(), '_sale_price_dates_to', '');
        $product->set_sale_price('');
        $product->save();
		
    }
		if ( class_exists( 'LiteSpeed_Cache_API' ) ) {
        // Purga total cuando WooCommerce elimina sales vencidas
        do_action( 'litespeed_purge_all' );
    }
		
}

	// --- 2. Filtro de precio para loops / shortcodes ---
public function fg_filtrar_precio_oferta_vencida($precio, $product) {
    $sale_to = $product->get_date_on_sale_to();
    if ( $sale_to && $sale_to->getTimestamp() < time() ) {	
		do_action('woocommerce_scheduled_sales');
        // Oferta vencida → devolver precio regular
        return $product->get_regular_price();
    }	
    return $precio;
}
	
	
	

	
}

new FG_Gestion_Stock_Precios_Ofertas();

function fg_is_trial_active() {
    $start = intval(get_option('fg_trial_start'));
    if (!$start) return false;

    $now = current_time('timestamp');
    $dias = 15 * DAY_IN_SECONDS;

    return ($now - $start) < $dias;
}


register_activation_hook(__FILE__, 'fg_plugin_activate');

function fg_plugin_activate() {
    if (!get_option('fg_trial_start')) {        
		add_option('fg_trial_start', current_time('timestamp'));
    }
}